<?php
require "conn.php";

// Handle Delete
if (isset($_GET['delete'])) {
    $agent_id = intval($_GET['delete']);
    $delete_query = "DELETE FROM agents WHERE agent_id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $agent_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success_msg = "Agent deleted successfully!";
    } else {
        $error_msg = "Error deleting agent: " . mysqli_error($conn);
    }
    mysqli_stmt_close($stmt);
}

// Handle Insert/Update
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $agent_id = isset($_POST['agent_id']) ? intval($_POST['agent_id']) : 0;
    $agent_name = trim($_POST['agent_name']);
    $agent_cnic = trim($_POST['agent_cnic']);
    $agent_cell = trim($_POST['agent_cell']);
    $recipient_province = trim($_POST['recipient_province']);
    $recipient_division = trim($_POST['recipient_division']);
    $recipient_district = trim($_POST['recipient_district']);
    $status = $_POST['status'];
    
    // Validation
    if (empty($agent_name) || empty($agent_cnic) || empty($agent_cell) || empty($recipient_province) || empty($recipient_division) || empty($recipient_district)) {
        $error_msg = "All fields are required!";
    } elseif (!preg_match('/^\d{5}-\d{7}-\d{1}$/', $agent_cnic)) {
        $error_msg = "CNIC format must be: 00000-0000000-0";
    } elseif (!preg_match('/^0\d{3}-\d{7}$/', $agent_cell)) {
        $error_msg = "Cell number format must be: 0000-0000000";
    } else {
        if ($agent_id > 0) {
            // Update
            $update_query = "UPDATE agents SET agent_name = ?, agent_cnic = ?, agent_cell = ?, recipient_province = ?, recipient_division = ?, recipient_district = ?, status = ? WHERE agent_id = ?";
            $stmt = mysqli_prepare($conn, $update_query);
            mysqli_stmt_bind_param($stmt, "sssssssi", $agent_name, $agent_cnic, $agent_cell, $recipient_province, $recipient_division, $recipient_district, $status, $agent_id);
            
            if (mysqli_stmt_execute($stmt)) {
                $success_msg = "Agent updated successfully!";
            } else {
                if (mysqli_errno($conn) == 1062) {
                    $error_msg = "CNIC already exists!";
                } else {
                    $error_msg = "Error updating agent: " . mysqli_error($conn);
                }
            }
        } else {
            // Insert
            $insert_query = "INSERT INTO agents (agent_name, agent_cnic, agent_cell, recipient_province, recipient_division, recipient_district, status) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt = mysqli_prepare($conn, $insert_query);
            mysqli_stmt_bind_param($stmt, "sssssss", $agent_name, $agent_cnic, $agent_cell, $recipient_province, $recipient_division, $recipient_district, $status);
            
            if (mysqli_stmt_execute($stmt)) {
                $success_msg = "Agent added successfully!";
            } else {
                if (mysqli_errno($conn) == 1062) {
                    $error_msg = "CNIC already exists!";
                } else {
                    $error_msg = "Error adding agent: " . mysqli_error($conn);
                }
            }
        }
        mysqli_stmt_close($stmt);
    }
}

// Fetch agent for editing
$edit_agent = null;
if (isset($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    $edit_query = "SELECT * FROM agents WHERE agent_id = ?";
    $stmt = mysqli_prepare($conn, $edit_query);
    mysqli_stmt_bind_param($stmt, "i", $edit_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $edit_agent = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);
}

// Fetch all agents
$agents_query = "SELECT * FROM agents ORDER BY agent_id DESC";
$agents_result = mysqli_query($conn, $agents_query);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>FDIMS :: Agents Management</title>
    <link rel="shortcut icon" href="assets/images/favicon.png">
    <link href="https://fonts.googleapis.com/css?family=Roboto:400,500,700&display=swap" rel="stylesheet">
    <link href="font-awesome/4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <link href="assets/css/bootstrap.min.css" rel="stylesheet" type="text/css">
    <link href="assets/css/style.css" rel="stylesheet" type="text/css">
    <link href="assets/css/custom.css" rel="stylesheet" type="text/css">
    <style>
        body {
            background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container-custom {
            max-width: 1400px;
            margin: 0 auto;
            background: #fff;
            padding: 35px;
            border-radius: 12px;
            box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        }
        .page-header {
            border-bottom: 3px solid #28a745;
            padding-bottom: 20px;
            margin-bottom: 35px;
        }
        .page-header h2 {
            color: #2c3e50;
            font-weight: 600;
            margin-bottom: 5px;
        }
        .alert {
            padding: 15px 20px;
            margin-bottom: 25px;
            border-radius: 6px;
            border-left: 4px solid;
        }
        .alert-success {
            background-color: #d4edda;
            border-left-color: #28a745;
            color: #155724;
        }
        .alert-danger {
            background-color: #f8d7da;
            border-left-color: #dc3545;
            color: #721c24;
        }
        .form-section {
            background: linear-gradient(to bottom, #f8f9fa 0%, #e9ecef 100%);
            padding: 30px;
            border-radius: 8px;
            margin-bottom: 40px;
            border: 1px solid #dee2e6;
        }
        .form-section h4 {
            color: #495057;
            font-weight: 600;
            margin-bottom: 25px;
            padding-bottom: 10px;
            border-bottom: 2px solid #28a745;
        }
        .table-section {
            margin-top: 40px;
        }
        .table-section h4 {
            color: #495057;
            font-weight: 600;
            margin-bottom: 20px;
        }
        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .table {
            margin-bottom: 0;
            font-size: 14px;
        }
        .table thead th {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border: none;
            padding: 15px 12px;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 12px;
            letter-spacing: 0.5px;
            vertical-align: middle;
            text-align: center;
        }
        .table tbody td {
            padding: 12px;
            vertical-align: middle;
            border-color: #e9ecef;
            text-align: center;
        }
        .table tbody tr {
            transition: all 0.3s ease;
        }
        .table tbody tr:hover {
            background-color: #f1f3f5;
            transform: scale(1.01);
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        .btn-action {
            padding: 6px 14px;
            font-size: 12px;
            font-weight: 500;
            border-radius: 4px;
            transition: all 0.3s ease;
            border: none;
            min-width: 70px;
            display: inline-block;
            text-align: center;
        }
        .btn-action i {
            margin-right: 4px;
        }
        .btn-action:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .btn-sm.btn-info {
            background-color: #17a2b8;
            color: white;
        }
        .btn-sm.btn-info:hover {
            background-color: #138496;
        }
        .btn-sm.btn-danger {
            background-color: #dc3545;
            color: white;
        }
        .btn-sm.btn-danger:hover {
            background-color: #c82333;
        }
        .badge {
            padding: 6px 12px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .badge-success {
            background-color: #28a745;
            color: white;
        }
        .badge-secondary {
            background-color: #6c757d;
            color: white;
        }
        .btn-success {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            border: none;
            padding: 10px 25px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(40,167,69,0.3);
        }
        .btn-secondary {
            background-color: #6c757d;
            border: none;
            padding: 10px 25px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-secondary:hover {
            background-color: #5a6268;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(108,117,125,0.3);
        }
        .form-control {
            border-radius: 4px;
            border: 1px solid #ced4da;
            padding: 8px 12px;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #28a745;
            box-shadow: 0 0 0 0.2rem rgba(40,167,69,0.25);
        }
        .format-hint {
            font-size: 12px;
            color: #6c757d;
            margin-top: 3px;
        }
        .action-buttons {
            white-space: nowrap;
        }
    </style>
</head>
<body>
    <div class="container-custom">
        <div class="page-header">
            <h2><i class="fa fa-users"></i> Agents Management</h2>
            <p class="text-muted">Add, Edit, and Delete Agents</p>
        </div>

        <?php if (isset($success_msg)): ?>
            <div class="alert alert-success">
                <i class="fa fa-check-circle"></i> <?php echo htmlspecialchars($success_msg); ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error_msg)): ?>
            <div class="alert alert-danger">
                <i class="fa fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_msg); ?>
            </div>
        <?php endif; ?>

        <!-- Form Section -->
        <div class="form-section">
            <h4><?php echo $edit_agent ? 'Edit Agent' : 'Add New Agent'; ?></h4>
            <form method="POST" action="agents_manage.php" id="agentForm">
                <?php if ($edit_agent): ?>
                    <input type="hidden" name="agent_id" value="<?php echo $edit_agent['agent_id']; ?>">
                <?php endif; ?>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Agent Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="agent_name" id="agent_name"
                                   value="<?php echo $edit_agent ? htmlspecialchars($edit_agent['agent_name']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Agent CNIC <span class="text-danger">*</span></label>
                            <input type="text" class="form-control mask_cnic" name="agent_cnic" id="agent_cnic"
                                   value="<?php echo $edit_agent ? htmlspecialchars($edit_agent['agent_cnic']) : ''; ?>" 
                                   placeholder="00000-0000000-0"
                                   required>
                            <div class="format-hint">Format: 00000-0000000-0</div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Agent Cell Number <span class="text-danger">*</span></label>
                            <input type="text" class="form-control mask_phone" name="agent_cell" id="agent_cell"
                                   value="<?php echo $edit_agent ? htmlspecialchars($edit_agent['agent_cell']) : ''; ?>" 
                                   placeholder="0000-0000000"
                                   required>
                            <div class="format-hint">Format: 0000-0000000</div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Recipient Province <span class="text-danger">*</span></label>
                            <select class="form-control" name="recipient_province" required>
                                <option value="">--Select Province--</option>
                                <option value="Punjab" <?php echo ($edit_agent && $edit_agent['recipient_province'] == 'Punjab') ? 'selected' : ''; ?>>Punjab</option>
                                <option value="Khyber Pakhtunkhwa" <?php echo ($edit_agent && $edit_agent['recipient_province'] == 'Khyber Pakhtunkhwa') ? 'selected' : ''; ?>>Khyber Pakhtunkhwa</option>
                                <option value="Sindh" <?php echo ($edit_agent && $edit_agent['recipient_province'] == 'Sindh') ? 'selected' : ''; ?>>Sindh</option>
                                <option value="Balochistan" <?php echo ($edit_agent && $edit_agent['recipient_province'] == 'Balochistan') ? 'selected' : ''; ?>>Balochistan</option>
                                <option value="Islamabad" <?php echo ($edit_agent && $edit_agent['recipient_province'] == 'Islamabad') ? 'selected' : ''; ?>>Islamabad</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Recipient Division <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="recipient_division" 
                                   value="<?php echo $edit_agent ? htmlspecialchars($edit_agent['recipient_division']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Recipient District <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="recipient_district" 
                                   value="<?php echo $edit_agent ? htmlspecialchars($edit_agent['recipient_district']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Status <span class="text-danger">*</span></label>
                            <select class="form-control" name="status" required>
                                <option value="Active" <?php echo ($edit_agent && $edit_agent['status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                <option value="Inactive" <?php echo ($edit_agent && $edit_agent['status'] == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <button type="submit" class="btn btn-success">
                        <i class="fa fa-save"></i> <?php echo $edit_agent ? 'Update Agent' : 'Add Agent'; ?>
                    </button>
                    <?php if ($edit_agent): ?>
                        <a href="agents_manage.php" class="btn btn-secondary">
                            <i class="fa fa-times"></i> Cancel
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <!-- Table Section -->
        <div class="table-section">
            <h4>All Agents</h4>
            <div class="table-responsive">
                <table class="table table-bordered table-hover">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Agent Name</th>
                            <th>CNIC</th>
                            <th>Cell Number</th>
                            <th>Recipient Location</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (mysqli_num_rows($agents_result) > 0): ?>
                            <?php while ($agent = mysqli_fetch_assoc($agents_result)): ?>
                                <tr>
                                    <td><?php echo $agent['agent_id']; ?></td>
                                    <td><?php echo htmlspecialchars($agent['agent_name']); ?></td>
                                    <td><?php echo htmlspecialchars($agent['agent_cnic']); ?></td>
                                    <td><?php echo htmlspecialchars($agent['agent_cell']); ?></td>
                                    <td><?php echo htmlspecialchars($agent['recipient_district']) . ', ' . htmlspecialchars($agent['recipient_division']); ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $agent['status'] == 'Active' ? 'success' : 'secondary'; ?>">
                                            <?php echo $agent['status']; ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d-m-Y', strtotime($agent['created_at'])); ?></td>
                                    <td class="action-buttons">
                                        <a href="agents_manage.php?edit=<?php echo $agent['agent_id']; ?>" 
                                           class="btn btn-sm btn-info btn-action"
                                           title="Edit Agent">
                                            <i class="fa fa-edit"></i> Edit
                                        </a>
                                        <a href="agents_manage.php?delete=<?php echo $agent['agent_id']; ?>" 
                                           class="btn btn-sm btn-danger btn-action"
                                           title="Delete Agent"
                                           onclick="return confirm('Are you sure you want to delete this agent?');">
                                            <i class="fa fa-trash"></i> Delete
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="8" class="text-center">No agents found. Add your first agent above.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="assets/js/jquery.min.js"></script>
    <script src="assets/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/jquery.mask.js"></script>
    
    <script>
        $(document).ready(function() {
            // CNIC Mask: 00000-0000000-0
            $('.mask_cnic').mask('00000-0000000-0');
            
            // Phone Mask: 0000-0000000
            $('.mask_phone').mask('0000-0000000');
            
            // Focus on first field
            $('#agent_name').focus();
        });
    </script>
</body>
</html>
