<?php
require "conn.php";

// Handle Delete
if (isset($_GET['delete'])) {
    $driver_id = intval($_GET['delete']);
    $delete_query = "DELETE FROM drivers WHERE driver_id = ?";
    $stmt = mysqli_prepare($conn, $delete_query);
    mysqli_stmt_bind_param($stmt, "i", $driver_id);
    
    if (mysqli_stmt_execute($stmt)) {
        $success_msg = "Driver deleted successfully!";
    } else {
        $error_msg = "Error deleting driver: " . mysqli_error($conn);
    }
    mysqli_stmt_close($stmt);
}

// Handle Insert/Update
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $driver_id = isset($_POST['driver_id']) ? intval($_POST['driver_id']) : 0;
    $driver_name = trim($_POST['driver_name']);
    $driver_cnic = trim($_POST['driver_cnic']);
    $driver_mobno = trim($_POST['driver_mobno']);
    $status = $_POST['status'];
    
    // Validation
    if (empty($driver_name) || empty($driver_cnic) || empty($driver_mobno)) {
        $error_msg = "All fields are required!";
    } elseif (!preg_match('/^\d{5}-\d{7}-\d{1}$/', $driver_cnic)) {
        $error_msg = "CNIC format must be: 00000-0000000-0";
    } elseif (!preg_match('/^0\d{3}-\d{7}$/', $driver_mobno)) {
        $error_msg = "Mobile number format must be: 0000-0000000";
    } else {
        if ($driver_id > 0) {
            // Update
            $update_query = "UPDATE drivers SET driver_name = ?, driver_cnic = ?, driver_mobno = ?, status = ? WHERE driver_id = ?";
            $stmt = mysqli_prepare($conn, $update_query);
            mysqli_stmt_bind_param($stmt, "ssssi", $driver_name, $driver_cnic, $driver_mobno, $status, $driver_id);
            
            if (mysqli_stmt_execute($stmt)) {
                $success_msg = "Driver updated successfully!";
            } else {
                if (mysqli_errno($conn) == 1062) {
                    $error_msg = "CNIC already exists!";
                } else {
                    $error_msg = "Error updating driver: " . mysqli_error($conn);
                }
            }
        } else {
            // Insert
            $insert_query = "INSERT INTO drivers (driver_name, driver_cnic, driver_mobno, status) VALUES (?, ?, ?, ?)";
            $stmt = mysqli_prepare($conn, $insert_query);
            mysqli_stmt_bind_param($stmt, "ssss", $driver_name, $driver_cnic, $driver_mobno, $status);
            
            if (mysqli_stmt_execute($stmt)) {
                $success_msg = "Driver added successfully!";
            } else {
                if (mysqli_errno($conn) == 1062) {
                    $error_msg = "CNIC already exists!";
                } else {
                    $error_msg = "Error adding driver: " . mysqli_error($conn);
                }
            }
        }
        mysqli_stmt_close($stmt);
    }
}

// Fetch driver for editing
$edit_driver = null;
if (isset($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    $edit_query = "SELECT * FROM drivers WHERE driver_id = ?";
    $stmt = mysqli_prepare($conn, $edit_query);
    mysqli_stmt_bind_param($stmt, "i", $edit_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $edit_driver = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);
}

// Fetch all drivers
$drivers_query = "SELECT * FROM drivers ORDER BY driver_id DESC";
$drivers_result = mysqli_query($conn, $drivers_query);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>FDIMS :: Drivers Management</title>
    <link rel="shortcut icon" href="assets/images/favicon.png">
    <link href="https://fonts.googleapis.com/css?family=Roboto:400,500,700&display=swap" rel="stylesheet">
    <link href="font-awesome/4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <link href="assets/css/bootstrap.min.css" rel="stylesheet" type="text/css">
    <link href="assets/css/style.css" rel="stylesheet" type="text/css">
    <link href="assets/css/custom.css" rel="stylesheet" type="text/css">
    <style>
        body {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            padding: 20px;
            min-height: 100vh;
        }
        .container-custom {
            max-width: 1400px;
            margin: 0 auto;
            background: #fff;
            padding: 35px;
            border-radius: 12px;
            box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        }
        .page-header {
            border-bottom: 3px solid #f5576c;
            padding-bottom: 20px;
            margin-bottom: 35px;
        }
        .page-header h2 {
            color: #2c3e50;
            font-weight: 600;
            margin-bottom: 5px;
        }
        .alert {
            padding: 15px 20px;
            margin-bottom: 25px;
            border-radius: 6px;
            border-left: 4px solid;
        }
        .alert-success {
            background-color: #d4edda;
            border-left-color: #28a745;
            color: #155724;
        }
        .alert-danger {
            background-color: #f8d7da;
            border-left-color: #dc3545;
            color: #721c24;
        }
        .form-section {
            background: linear-gradient(to bottom, #f8f9fa 0%, #e9ecef 100%);
            padding: 30px;
            border-radius: 8px;
            margin-bottom: 40px;
            border: 1px solid #dee2e6;
        }
        .form-section h4 {
            color: #495057;
            font-weight: 600;
            margin-bottom: 25px;
            padding-bottom: 10px;
            border-bottom: 2px solid #f5576c;
        }
        .table-section {
            margin-top: 40px;
        }
        .table-section h4 {
            color: #495057;
            font-weight: 600;
            margin-bottom: 20px;
        }
        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .table {
            margin-bottom: 0;
            font-size: 14px;
        }
        .table thead th {
            background: linear-gradient(135deg, #f5576c 0%, #f093fb 100%);
            color: white;
            border: none;
            padding: 15px 12px;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 12px;
            letter-spacing: 0.5px;
            vertical-align: middle;
            text-align: center;
        }
        .table tbody td {
            padding: 12px;
            vertical-align: middle;
            border-color: #e9ecef;
            text-align: center;
        }
        .table tbody tr {
            transition: all 0.3s ease;
        }
        .table tbody tr:hover {
            background-color: #f1f3f5;
            transform: scale(1.01);
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        .btn-action {
            padding: 6px 14px;
            font-size: 12px;
            font-weight: 500;
            border-radius: 4px;
            transition: all 0.3s ease;
            border: none;
            min-width: 70px;
            display: inline-block;
            text-align: center;
        }
        .btn-action i {
            margin-right: 4px;
        }
        .btn-action:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .btn-sm.btn-info {
            background-color: #17a2b8;
            color: white;
        }
        .btn-sm.btn-info:hover {
            background-color: #138496;
        }
        .btn-sm.btn-danger {
            background-color: #dc3545;
            color: white;
        }
        .btn-sm.btn-danger:hover {
            background-color: #c82333;
        }
        .badge {
            padding: 6px 12px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .badge-success {
            background-color: #28a745;
            color: white;
        }
        .badge-secondary {
            background-color: #6c757d;
            color: white;
        }
        .btn-warning {
            background: linear-gradient(135deg, #f5576c 0%, #f093fb 100%);
            border: none;
            padding: 10px 25px;
            font-weight: 500;
            transition: all 0.3s ease;
            color: white;
        }
        .btn-warning:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(245,87,108,0.3);
            color: white;
        }
        .btn-secondary {
            background-color: #6c757d;
            border: none;
            padding: 10px 25px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-secondary:hover {
            background-color: #5a6268;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(108,117,125,0.3);
        }
        .form-control {
            border-radius: 4px;
            border: 1px solid #ced4da;
            padding: 8px 12px;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #f5576c;
            box-shadow: 0 0 0 0.2rem rgba(245,87,108,0.25);
        }
        .format-hint {
            font-size: 12px;
            color: #6c757d;
            margin-top: 3px;
        }
        .action-buttons {
            white-space: nowrap;
        }
    </style>
</head>
<body>
    <div class="container-custom">
        <div class="page-header">
            <h2><i class="fa fa-car"></i> Drivers Management</h2>
            <p class="text-muted">Add, Edit, and Delete Drivers</p>
        </div>

        <?php if (isset($success_msg)): ?>
            <div class="alert alert-success">
                <i class="fa fa-check-circle"></i> <?php echo htmlspecialchars($success_msg); ?>
            </div>
        <?php endif; ?>

        <?php if (isset($error_msg)): ?>
            <div class="alert alert-danger">
                <i class="fa fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_msg); ?>
            </div>
        <?php endif; ?>

        <!-- Form Section -->
        <div class="form-section">
            <h4><?php echo $edit_driver ? 'Edit Driver' : 'Add New Driver'; ?></h4>
            <form method="POST" action="drivers_manage.php" id="driverForm">
                <?php if ($edit_driver): ?>
                    <input type="hidden" name="driver_id" value="<?php echo $edit_driver['driver_id']; ?>">
                <?php endif; ?>
                
                <div class="row">
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Driver Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="driver_name" id="driver_name"
                                   value="<?php echo $edit_driver ? htmlspecialchars($edit_driver['driver_name']) : ''; ?>" 
                                   required>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Driver CNIC <span class="text-danger">*</span></label>
                            <input type="text" class="form-control mask_cnic" name="driver_cnic" id="driver_cnic"
                                   value="<?php echo $edit_driver ? htmlspecialchars($edit_driver['driver_cnic']) : ''; ?>" 
                                   placeholder="00000-0000000-0"
                                   required>
                            <div class="format-hint">Format: 00000-0000000-0</div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label>Driver Mobile Number <span class="text-danger">*</span></label>
                            <input type="text" class="form-control mask_phone" name="driver_mobno" id="driver_mobno"
                                   value="<?php echo $edit_driver ? htmlspecialchars($edit_driver['driver_mobno']) : ''; ?>" 
                                   placeholder="0000-0000000"
                                   required>
                            <div class="format-hint">Format: 0000-0000000</div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label>Status <span class="text-danger">*</span></label>
                            <select class="form-control" name="status" required>
                                <option value="Active" <?php echo ($edit_driver && $edit_driver['status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                <option value="Inactive" <?php echo ($edit_driver && $edit_driver['status'] == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                            </select>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <button type="submit" class="btn btn-warning">
                        <i class="fa fa-save"></i> <?php echo $edit_driver ? 'Update Driver' : 'Add Driver'; ?>
                    </button>
                    <?php if ($edit_driver): ?>
                        <a href="drivers_manage.php" class="btn btn-secondary">
                            <i class="fa fa-times"></i> Cancel
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>

        <!-- Table Section -->
        <div class="table-section">
            <h4>All Drivers</h4>
            <div class="table-responsive">
                <table class="table table-bordered table-hover">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Driver Name</th>
                            <th>CNIC</th>
                            <th>Mobile Number</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (mysqli_num_rows($drivers_result) > 0): ?>
                            <?php while ($driver = mysqli_fetch_assoc($drivers_result)): ?>
                                <tr>
                                    <td><?php echo $driver['driver_id']; ?></td>
                                    <td><?php echo htmlspecialchars($driver['driver_name']); ?></td>
                                    <td><?php echo htmlspecialchars($driver['driver_cnic']); ?></td>
                                    <td><?php echo htmlspecialchars($driver['driver_mobno']); ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $driver['status'] == 'Active' ? 'success' : 'secondary'; ?>">
                                            <?php echo $driver['status']; ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d-m-Y', strtotime($driver['created_at'])); ?></td>
                                    <td class="action-buttons">
                                        <a href="drivers_manage.php?edit=<?php echo $driver['driver_id']; ?>" 
                                           class="btn btn-sm btn-info btn-action"
                                           title="Edit Driver">
                                            <i class="fa fa-edit"></i> Edit
                                        </a>
                                        <a href="drivers_manage.php?delete=<?php echo $driver['driver_id']; ?>" 
                                           class="btn btn-sm btn-danger btn-action"
                                           title="Delete Driver"
                                           onclick="return confirm('Are you sure you want to delete this driver?');">
                                            <i class="fa fa-trash"></i> Delete
                                        </a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7" class="text-center">No drivers found. Add your first driver above.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="assets/js/jquery.min.js"></script>
    <script src="assets/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/jquery.mask.js"></script>
    
    <script>
        $(document).ready(function() {
            // CNIC Mask: 00000-0000000-0
            $('.mask_cnic').mask('00000-0000000-0');
            
            // Phone Mask: 0000-0000000
            $('.mask_phone').mask('0000-0000000');
            
            // Focus on first field
            $('#driver_name').focus();
        });
    </script>
</body>
</html>
